/* socket.h
 * Based on James Pee's article 'Guidelines for Wrapping Sockets in Classes'
 * published in the C/C++ Users journal.
 * Copyright (C) 2003 Michel Leunen
 * http://www.leunen.com/
 * michel@leunen.com
 *
 * This software is freeware and is provided at no charge to the user.
 * This software is provided 'as-is', without any express or implied warranty.
 * To the maximum extent permitted by applicable law will the author NOT be held
 * liable for any damages whatsoever arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose, including
 * commercial applications, and to freely redistribute it, with the following
 * restrictions:
 *
 * 1. If you use this software in a product, an acknowledgment in the product
 *    documentation would be appreciated.
 *
 * 2. If you use this software in a commercial application, please send me an email
 *	  detailing the application.
 *
 * 3. This notice may not be removed or altered from any source distribution.
 *
 * 4. It may be distributed by any means, provided that the original files
 * 		as supplied by the author remain intact and no charge is made other than
 * 		for reasonable distribution costs. If you are a vendor, if you want to
 * 		include this software in a compilation or in a magazine cover disk
 * 		you HAVE TO CONTACT ME to obtain my authorization before distribution.
 */

#ifndef SOCKET_H
#define SOCKET_H

#include <string>
#include <iostream>
#include <exception>
#include <winsock2.h>

namespace mlLib
{
//
// Exception class used by all classes
//

class socket_exception: public std::exception
{
  public:
		explicit socket_exception(const std::string &what);
		virtual const char* what()const throw();
	private:
		std::string msg_;
};

//
// This class is responsible for initializing winsock
//

class Winsock
{
	public:
		Winsock();
		virtual ~Winsock();
	private:
		static int refcount_;
};

//
// Utility classes
//

struct LocalHost: public Winsock
{
  LocalHost(){}
  std::string GetHostName()const;
  std::string GetIPAddress()const;
};

class Host: public Winsock
{
	public:
		Host(const std::string& address);
		
		bool IsDotted()const;
		std::string GetHostName()const;
		std::string GetIPAddress()const;
	private:
		std::string Address_;
};

//
// Base class
//

class Socket: public Winsock
{
	public:
    Socket();
  	Socket(int domain,int type,int protocol);
    // Copy constructor and assignment operator transfer ownership
    // of the socket to the new Socket object.
    Socket(Socket& rhs);
    Socket& operator=(Socket& rhs);
  	virtual ~Socket();

  	void Bind(unsigned short port);
  	Socket Accept();
  	void Listen(int connections);
  	void Connect(const std::string& host,unsigned short port);
  	void Close();
  	unsigned long BytesAvailable();
    bool Readable();
    
    //TCP read and write functions
    int Read(void* data,int size);
  	void Write(const void* data,int size);
    std::string ReadLine();
    void WriteString(const std::string& data);

	protected:
    bool Initialized;
  	int Domain;
    int SocketDescriptor;
    bool GetAddress(const std::string& host,unsigned short port,sockaddr_in* in);
    //UDP read and write functions
    int RecvFrom(int desc,void* data,int size,sockaddr* source);
    void SendTo(int desc,const void* data,int size,sockaddr* target);

  private:
  	std::string ClientIP;
    Socket(int,const std::string&);
    int ReleaseOwnership();
};

//
// TCP Client
//

enum Protocol {prTCP,prUDP};

class TCPClient : public Socket
{
  public:
    TCPClient(const std::string& host,unsigned short port);
};

//
// TCP Server
//

class TCPServer : public Socket
{
  public:
    TCPServer(unsigned short port,int connections = 10);
    Socket AcceptClient();
};

//
// UDP Client
//

class UDPClient : public Socket
{
  public:
    UDPClient(const std::string& host,unsigned short port);
    int ReadFrom(void* data,int size);
    void WriteTo(const void* data,int size);

  private:
    sockaddr RemoteAddress;
    void ResolveAddress(const std::string& host,unsigned short port);
};

//
// UDP Server
//

class UDPServer : public Socket
{
  public:
    UDPServer(unsigned short port);
    int ReadFrom(void* data,int size);
    void WriteTo(const void* data,int size);
    
  private:
    sockaddr ClientAddress;
};

} //end namespace mlLib
#endif
